C      *****************************************************************
C      * Subroutine CalcSpeciesConc                                    *
C      * Returns the SpConc array (species concentration) for each     *
C      * dependent species based on the concentration of each component*
C      * and the equilibrium constants.  Concentrations of some        *
C      * components are determined by the presence of solid and        *
C      * gaseous phases.                                               *
C      *                                                               *
C      * Variables:                                                    *
C      * C, C1     -Local - Component index.  Unitless.                *
C      * CBal      -Output- Ionic charge balance.  Unitless.           *
C      * Gamma     -Local - A vector, of size NSpecies, of activity    *
C      *                    coefficients.  Units are L/mol.            *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * IStr      -Output- Ionic strength.  Units are mol/L.  (These  *
C      *                    units do not result from the calculation   *
C      *                    but are defined by usage (Morel 1983).)    *
C      * LNum      -Input - The current layer number.  Unitless.       *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NPhases   -Input - The number of phases in the system.        *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * NSpecies  -Input - The number of species in the system (i.e.  *
C      *                    the number of components plus the number   *
C      *                    of dependent species).  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *
C      * P         -Local - Phase array index.  Unitless.              *
C      * Q         -Local - Phase array index.  Unitless.              *
C      * T1        -Local - Temporary work space.  Units consistent    *
C      *                    with items stored.                         *
C      * tEqCnst   -Input - A matrix, of size NSpecies by NLayers, of  *
C      *                    the transformed equilibrium constants.  The*
C      *                    units depend on stochiometric coefficients *
C      *                    (SC) and the units of species              *
C      *                    concentrations (SpConc) (mol/L).           *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * tSC       -Input - A matrix, of size NSpecies by NComp, of    *
C      *                    the transformed stoichiometric coefficients*
C      *                    of each comp for each species.  Unitless.  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * S         -Local - Species array index.  Unitless.            *
C      * SComp     -Input - A vector, of size NPhases, of the          *
C      *                    component associated with each phase.      *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SOrder    -Input - A vector, of size NPhases, of order in     *
C      *                    which to consider phases in mass action    *
C      *                    substitutions.  Unitless.                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SpConc    -In/Out- A matrix, of size NSpecies by NLayers, of  *
C      *                    species concentrations, one set for each   *
C      *                    layer. Units are mol/L.                    *
C      *                    (Common block VModel, file VModel.f)       *
C      * Species   -Local - Species index for SC array.  Unitless.     *
C      *****************************************************************
       SUBROUTINE CALCSPECIESCONC(LNUM, GOAHEAD)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							INTEGER C, C1, GOAHEAD, LNUM, Q, P, S
       REAL*8 CBAL, GAMMA(NSPECIESSIZE), ISTR, T1

C      *-------------------------------------------------------*
C      * Initialize flag to indicate that there are no errors. *
C      *-------------------------------------------------------*
       GOAHEAD = 1

C      *------------------------------------------------*
C      * Get the ionic strength and the charge balance. *
C      *------------------------------------------------*
       CALL CHARGEBAL(LNUM, ISTR, CBAL)

C      *--------------------------------------*
C      * Calculate the activity coefficients. *
C      *--------------------------------------*
       CALL CALCACTCOEFF(LNUM, ISTR, GOAHEAD, GAMMA)
       IF (GOAHEAD.NE.1) GOTO 9999

C      *-----------------------------------------------------------*
C      * Substitute solubility expression for selected components. *
C      *-----------------------------------------------------------*
       DO 110 Q=1, NPHASES
          P = SORDER(Q)
          C1 = SCOMP(P)
          T1 = 1
          DO 100 C=1, NCOMP
             IF (C.NE.C1) THEN
                T1 = T1 * (SPCONC(C,LNUM) * GAMMA(C)) ** TSC(C1,C)
             ENDIF
  100     CONTINUE
          SPCONC(C1,LNUM) = T1 * TEQCNST(C1,LNUM) / GAMMA(C1)
  110  CONTINUE

C      *----------------------------------------------------------*
C      * It is assumed that the species and the components are in *
C      * equilibrium.  Thus, the activity of a dependent species  *
C      * is a function of the activity of each constituent        *
C      * component and the equilibrium constant.                  *                                             
C      *----------------------------------------------------------*
       DO 130 S=NCOMP+1, NSPECIES
          T1 = 1
          DO 120 C=1, NCOMP
             T1 = T1 * (SPCONC(C,LNUM) * GAMMA(C))  ** TSC(S,C)
  120     CONTINUE
          SPCONC(S,LNUM) = T1 * TEQCNST(S,LNUM) / GAMMA(S)
  130  CONTINUE

C      *---------------*
C      * Escape hatch. *
C      *---------------*
 9999  CONTINUE

       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
